/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package rdc;

import java.net.URL;

import com.inet.report.Area;
import com.inet.report.Element;
import com.inet.report.Engine;
import com.inet.report.Field;
import com.inet.report.FieldElement;
import com.inet.report.FieldPart;
import com.inet.report.Fields;
import com.inet.report.FontProperties;
import com.inet.report.FormulaField;
import com.inet.report.Paragraph;
import com.inet.report.RDC;
import com.inet.report.Section;
import com.inet.report.Text;
import com.inet.report.TextPart;

/**
 * This example shows how to load an existing report from file to change some values of text fields.
 * @see Text
 * @see Paragraph
 * @see TextPart
 * @see FieldPart
 * @see com.inet.report.Section#addText(int, int, int, int)
 * @see com.inet.report.Section#getElementsV()
 * @see com.inet.report.Section#getElements()
 */
public class TextSample extends RDCSample {

    static final String MAINREPORT = "samples/rdc/mainrep.rpt";

    /**
     * Change property values of elements of an existing report.
     * @param exportFmt the output format (e.g. Engine.EXPORT_PDF)
     * @return the new engine for the report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            //create a new Engine
            Engine eng = new Engine( exportFmt );
            //set the name of the report file
            URL reportURL = getClass().getResource( '/' + MAINREPORT );
            if( reportURL != null ) {
                eng.setReportFile( reportURL );
            } else {
                eng.setReportFile( MAINREPORT );
            }

            //hange the properties of a text element (label)
            Area area = eng.getArea( "PH" );//Page Header Area
            Section section = area.getSection( 0 );//First Section of Page Header Area

            Text text = null;
            Paragraph para = null;
            TextPart tPart = null;
            
            //Looking for the wanted text. Use properties to distinguish the Text you need from others.
            for( Element elem : section.getElements() ) {
                if( elem instanceof Text ) {
                    text = (Text)elem;
                    //The first line of the Text
                    para = text.getParagraph( 0 );
                    if( para.getPart( 0 ) instanceof TextPart ) {
                        //The first TextPart of the line
                        tPart = (TextPart)para.getPart( 0 );
                        if( tPart.getText().equals( "ContactName" ) ) {
                            break;
                        }
                    }
                }
            }
            //Set some properties
            if( tPart != null ) {
                tPart.setText( "Name" );
                tPart.setBold( false );
                tPart.setUnderline( false );
                tPart.setFontColor( RDC.COLOR_BLUE );
                tPart.setFontSize( 12 );
                tPart.setItalic( true );
                tPart.setFontName( FontProperties.FONT_SANS_SERIF );
            }

            /*
             * Change a property of a single values of a FieldElement
             */

            //get Detail Area and first Detail Section
            area = eng.getArea( "D" );
            section = area.getSection( 0 );

            //looking for the FieldElement for 'Customer.Contact Last Name' database field.
            for( Element elem : section.getElements() ) {
                if( elem instanceof FieldElement ) {
                    FieldElement fElem = (FieldElement)elem;
                    Field field = fElem.getField();
                    if( field.getName().equals( "Customers.ContactName" ) ) {
                        //add new FormulaField
                        Fields fields = eng.getFields();
                        FormulaField fmField =
                            fields.addFormulaField( "font color", "if {Customers.ContactName} = \"Paul Stevens\" then blue else black",
                                                    FormulaField.FORMULA_PROPERTY );
                        //enable formula for setting the property 'font color' of this FieldElement
                        fElem.setFontColorFormula( fmField );
                    }
                }
            }

            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new TextSample().initUI();
    }
}
